/* Module:          SQLBindParameter.c
 *
 * Description:     Binds a buffer to a parameter marker in an SQL statement. 
 *
 * Classes:         
 *
 * API functions:   SQLBindParameter
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLBindParameter(
						   SQLHSTMT			hDrvStmt,
						   SQLUSMALLINT		nParameterNumber,
						   SQLSMALLINT		nIOType,
						   SQLSMALLINT		nBufferType,
						   SQLSMALLINT		nParamType,
						   SQLUINTEGER		nParamLength,
						   SQLSMALLINT		nScale,
						   SQLPOINTER		pData,
						   SQLINTEGER		nBufferLength,
						   SQLINTEGER		*pnLengthOrIndicator
						   )
{
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	static char *func="SQLBindParameter";

	mylog( "%s: entering...\n", func);

	if( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	if(stmt->parameters_allocated < nParameterNumber)
	{
		ParameterInfoClass *old_parameters;
		int i, old_parameters_allocated;

		old_parameters = stmt->parameters;
		old_parameters_allocated = stmt->parameters_allocated;

		stmt->parameters = (ParameterInfoClass *) malloc(sizeof(ParameterInfoClass)*(nParameterNumber));
		if ( ! stmt->parameters)
		{
			stmt->errornumber = STMT_NO_MEMORY_ERROR;
			stmt->errormsg = "Could not allocate memory for statement parameters";
			SC_log_error(func, "", stmt);
			return SQL_ERROR;
		}

		stmt->parameters_allocated = nParameterNumber;

		/* copy the old parameters over */
		for(i = 0; i < old_parameters_allocated; i++)
		{
			/* a structure copy should work */
			stmt->parameters[i] = old_parameters[i];
		}

		/* get rid of the old parameters, if there were any */
		if(old_parameters)
		{
			free(old_parameters);
		}
		
		/* zero out the newly allocated parameters (in case they skipped some, */
		/* so we don't accidentally try to use them later) */
		for(; i < stmt->parameters_allocated; i++)
		{
			stmt->parameters[i].buflen = 0;
			stmt->parameters[i].buffer = 0;
			stmt->parameters[i].used = 0;
			stmt->parameters[i].paramType = 0;
			stmt->parameters[i].CType = 0;
			stmt->parameters[i].SQLType = 0;
			stmt->parameters[i].precision = 0;
			stmt->parameters[i].scale = 0;
			stmt->parameters[i].data_at_exec = FALSE;
			stmt->parameters[i].lobj_oid = 0;
			stmt->parameters[i].EXEC_used = NULL;
			stmt->parameters[i].EXEC_buffer = NULL;
		}
	}

	nParameterNumber--;		/* use zero based column numbers for the below part */

	/* store the given info */
	stmt->parameters[nParameterNumber].buflen = nBufferLength;
	stmt->parameters[nParameterNumber].buffer = pData;
	stmt->parameters[nParameterNumber].used = pnLengthOrIndicator;
	stmt->parameters[nParameterNumber].paramType = nIOType;
	stmt->parameters[nParameterNumber].CType = nBufferType;
	stmt->parameters[nParameterNumber].SQLType = nParamType;
	stmt->parameters[nParameterNumber].precision = nParamLength;
	stmt->parameters[nParameterNumber].scale = nScale;

	/*	If rebinding a parameter that had data-at-exec stuff in it,
		then free that stuff
	*/
	if (stmt->parameters[nParameterNumber].EXEC_used)
	{
		free(stmt->parameters[nParameterNumber].EXEC_used);
		stmt->parameters[nParameterNumber].EXEC_used = NULL;
	}

	if (stmt->parameters[nParameterNumber].EXEC_buffer)
	{
		if (stmt->parameters[nParameterNumber].SQLType != SQL_LONGVARBINARY)
		{

			free(stmt->parameters[nParameterNumber].EXEC_buffer);
		}

		stmt->parameters[nParameterNumber].EXEC_buffer = NULL;
	}

	/*	Data at exec macro only valid for C char/binary data */
	if ((nParamType == SQL_LONGVARBINARY || nParamType == SQL_LONGVARCHAR)
		&& pnLengthOrIndicator && *pnLengthOrIndicator <= SQL_LEN_DATA_AT_EXEC_OFFSET)
	{

		stmt->parameters[nParameterNumber].data_at_exec = TRUE;
	}

	else
	{
		stmt->parameters[nParameterNumber].data_at_exec = FALSE;
	}

	mylog("SQLBindParamater: nParameterNumber=%d, paramType=%d, nBufferType=%d, nParamType=%d, nParamLength=%d, nScale=%d, pData=%d, *pnLengthOrIndicator = %d, data_at_exec = %d\n", nParameterNumber, nIOType, nBufferType, nParamType, nParamLength, nScale, pData, pnLengthOrIndicator ? *pnLengthOrIndicator: -777, stmt->parameters[nParameterNumber].data_at_exec);

	return SQL_SUCCESS;
}
